<?php

/**
 * Plugin Name: Cipher Core
 * Description: After install the Cipher WordPress Theme, you must need to install this "cipher-core" first to get all functions of Cipher WP Theme.
 * Plugin URI:  https://themeforest.net/user/inventxlab
 * Version:     1.0.0
 * Author:      inventxlab
 * Author URI:  https://inventxlab.com/
 * Text Domain: cipher-core
 * Elementor tested up to: 3.24.0
 * Elementor Pro tested up to: 3.24.0
 */

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly
}

if (! function_exists('get_file_data')) {
    require_once ABSPATH . 'wp-includes/functions.php';
}

$plugin_data = get_file_data(__FILE__, array('Version' => 'Version'), false);

if (site_url() == 'http://localhost/cipher') {
    define('CIPHER_CORE_VERSION', time());
} else {
    define('CIPHER_CORE_VERSION', $plugin_data['Version']);
}

if (!defined('CIPHER_DIR_PATH') || !defined('CIPHER_DIR_URL')) {
    define('CIPHER_DIR_PATH', plugin_dir_path(__FILE__));
    define('CIPHER_DIR_URL', plugin_dir_url(__FILE__));
}

final class Cipher_Core_Plugin
{

    private static $instance = null;
    private $dir;

    private function __construct()
    {
        $this->dir = plugin_dir_path(__FILE__);

        // Load textdomainmain for translations
        add_action('init', [$this, 'cipher_text_domain']);
        add_action('init', [$this, 'cipher_init_theme'], 5);

        // General hooks
        add_action('wp_enqueue_scripts', [$this, 'de_reg']);
        add_action('wp_enqueue_scripts', [$this, 'enqueue_styles']);
        add_action('admin_enqueue_scripts', [$this, 'admin_scripts']);

        // Elementor - only add Elementor related hooks when Elementor is active
        if ($this->is_elementor_active()) {
            add_action('elementor/frontend/after_register_styles', [$this, 'de_reg_font_awesome'], 20);
            add_action('elementor/frontend/after_enqueue_scripts', [$this, 'cipher_core_enqueue_editor_scripts']);
        }
    }

    // Initialize theme functions
    public function cipher_init_theme()
    {
        $this->load_required_files();
    }

    public static function instance()
    {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /*===Textdomaind Textdomain =====*/
    public function cipher_text_domain()
    {
        load_plugin_textdomain('cipher-core', false, CIPHER_DIR_PATH . '/languages');
    }

    /*====== Check Elementor availability =====*/
    private function is_elementor_active()
    {
        return did_action('elementor/loaded');
    }

    /*===== Elementor - Remove Font Awesome =====*/
    public function de_reg_font_awesome()
    {
        foreach (['solid', 'regular', 'brands'] as $style) {
            wp_deregister_style('elementor-icons-fa-' . $style);
        }
    }

    /*===== Deregister Elementor animations (optional) =====*/
    public function de_reg()
    {
        wp_deregister_style('e-animations');
    }

    /*===== Frontend scripts for Elementor widgets =====*/
    public function cipher_core_enqueue_editor_scripts()
    {
        wp_enqueue_script( 'cipher-core-elementor-editor-scripts', CIPHER_DIR_URL . 'assets/js/editor-scripts.js', ['jquery'], CIPHER_CORE_VERSION, true);
    }

    /*===== General frontend styles/scripts =====*/
    public function enqueue_styles()
    {
        wp_enqueue_style('cipher-core-plugin-style', CIPHER_DIR_URL . 'assets/css/style.css', [], CIPHER_CORE_VERSION, 'all');
    }

    /*===== Admin styles/scripts =====*/
    public function admin_scripts()
    {
        wp_enqueue_style('cipher-admin-style', CIPHER_DIR_URL . 'assets/css/admin.css', [], CIPHER_CORE_VERSION, 'all');
        wp_enqueue_media();
    }

    /*===== Load all required support files and options =====*/
    private function load_required_files()
    {
        $required_files = [
            'functions/inc/cipher-cpt.php',
            'functions/inc/cipher-helper.php',
            'functions/inc/svg-support.php',
            'functions/lib/codestar/codestar-framework.php',

            // Theme options
            'functions/theme-option/csf.php',
            'functions/theme-option/csf-body.php',
            'functions/theme-option/csf-header.php',
            'functions/theme-option/csf-breadcrumb.php',
            'functions/theme-option/csf-typography.php',
            'functions/theme-option/csf-blog.php',
            'functions/theme-option/csf-404.php',
            'functions/theme-option/csf-footer.php',
            'functions/theme-option/csf-custom-css.php',
            'functions/theme-option/csf-backup.php',

            // Metaboxes
            'functions/theme-option/metabox/csf-mt-post.php',
            'functions/theme-option/metabox/csf-mt-page.php',
            'functions/theme-option/metabox/csf-mt-header-footer.php',

            // Widgets
            'functions/wp-widgets/about-author.php',
            'functions/wp-widgets/contact-info.php',
            'functions/wp-widgets/posts.php',
        ];

        // Add Elementor init if Elementor is active
        if (did_action('elementor/loaded')) {
            $required_files[] = 'elementor/elementor-init.php';
        }

        // Load each required file
        foreach ($required_files as $file) {
            $path = $this->dir . ltrim($file, '/');
            if (file_exists($path)) {
                require_once $path;
            } else {
                error_log("[Cipher-Core] Required file not found: " . esc_html($file));
            }
        }
    }
}

// Initialize plugin
Cipher_Core_Plugin::instance();

/***** Load widgets Files*****/
function cipher_load_widget_files()
{

    $widget_files = [
        'functions/wp-widgets/widget-recent-posts.php',
        'functions/wp-widgets/widget-search.php',
        'functions/wp-widgets/widget-taxonomy.php',
        'functions/wp-widgets/widget-post-tags.php',
        'functions/wp-widgets/widget-about.php',
    ];

    foreach ($widget_files as $file) {
        $path = CIPHER_DIR_PATH . $file;

        if (file_exists($path)) {
            require_once $path;
        } else {
            error_log('[Cipher-Core] Missing widget file: ' . $file);
        }
    }
}
cipher_load_widget_files();

/***** Register widgets *****/
function cipher_register_widgets()
{
    $widgets = [
        'Cipher_Core_Recent_Post_Widget',
        'Cipher_Core_Search_Widget',
        'Cipher_Core_Post_Taxonomy',
        'Cipher_Core_Post_Tags_Widget',
        'Cipher_Core_About_Widget',
    ];

    foreach ($widgets as $widget) {
        if (class_exists($widget)) {
            register_widget($widget);
        } else {
            error_log("[Cipher-Core] Widget class not found: {$widget}");
        }
    }
}
add_action('widgets_init', 'cipher_register_widgets');
