<?php

/**
 * Plugin Name: Trackngo Core
 * Description: After install the Trackngo WordPress Theme, you must need to install this "trackngo-core" first to get all functions of Trackngo WP Theme.
 * Plugin URI:  https://themeforest.net/user/inventxlab
 * Version:     1.0.0
 * Author:      inventxlab
 * Author URI:  https://inventxlab.com/
 * Text Domain: trackngo-core
 * Elementor tested up to: 3.24.0
 * Elementor Pro tested up to: 3.24.0
 */

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly
}

if (! function_exists('get_file_data')) {
    require_once ABSPATH . 'wp-includes/functions.php';
}

$plugin_data = get_file_data(__FILE__, array('Version' => 'Version'), false);

if (site_url() == 'http://localhost/trackngo') {
    define('TRACKNGO_CORE_VERSION', time());
} else {
    define('TRACKNGO_CORE_VERSION', $plugin_data['Version']);
}

if (!defined('TRACKNGO_DIR_PATH') || !defined('TRACKNGO_DIR_URL')) {
    define('TRACKNGO_DIR_PATH', plugin_dir_path(__FILE__));
    define('TRACKNGO_DIR_URL', plugin_dir_url(__FILE__));
}

final class Trackngo_Core_Plugin
{

    private static $instance = null;
    private $dir;

    private function __construct()
    {
        $this->dir = plugin_dir_path(__FILE__);

        $this->dir = plugin_dir_path(__FILE__);

        define('TECHCO_DIR_PATH', plugin_dir_path(__FILE__));

        // Load textdomain for translations
        add_action('init', [$this, 'load_textdomain']);
        add_action('init', [$this, 'trackngo_init_theme'], 5);

        // General hooks
        add_action('wp_enqueue_scripts', [$this, 'de_reg']);
        add_action('wp_enqueue_scripts', [$this, 'enqueue_styles']);
        add_action('admin_enqueue_scripts', [$this, 'admin_scripts']);

        // Elementor - only add Elementor related hooks when Elementor is active
        if ($this->is_elementor_active()) {
            add_action('elementor/frontend/after_register_styles', [$this, 'de_reg_font_awesome'], 20);
            add_action('elementor/frontend/after_enqueue_scripts', [$this, 'trackngo_core_enqueue_frontend_scripts']);
        }
    }

    // Initialize theme functions
    public function trackngo_init_theme()
    {
        $this->load_required_files();
        $this->register_widgets();
    }

    public static function instance()
    {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /*===== Load Textdomain =====*/
    public function load_textdomain()
    {
        load_plugin_textdomain('trackngo-core', false, TRACKNGO_DIR_PATH . '/languages');
    }


    /*====== Check Elementor availability =====*/
    private function is_elementor_active()
    {
        return did_action('elementor/loaded');
    }

    /*===== Elementor - Remove Font Awesome =====*/
    public function de_reg_font_awesome()
    {
        foreach (['solid', 'regular', 'brands'] as $style) {
            wp_deregister_style('elementor-icons-fa-' . $style);
        }
    }

    /*===== Deregister Elementor animations (optional) =====*/
    public function de_reg()
    {
        wp_deregister_style('e-animations');
    }


    /*===== Frontend scripts for Elementor widgets =====*/
    public function trackngo_core_enqueue_frontend_scripts()
    {
        wp_enqueue_script('trackngo-core-elementor-editor-scripts', TRACKNGO_DIR_URL . 'assets/js/editor-scripts.js', ['jquery'], TRACKNGO_CORE_VERSION, true);
    }

    /*===== General frontend styles/scripts =====*/
    public function enqueue_styles()
    {
        wp_enqueue_style('trackngo-core-plugin-style', TRACKNGO_DIR_URL . 'assets/css/style.css', [], TRACKNGO_CORE_VERSION, 'all');
    }

    /*===== Admin styles/scripts =====*/
    public function admin_scripts()
    {
        wp_enqueue_style('trackngo-admin-style', TRACKNGO_DIR_URL . 'assets/css/admin.css', [], TRACKNGO_CORE_VERSION, 'all');
        wp_enqueue_media();
    }

    /*===== Register WordPress widgets =====*/
    public function register_widgets() {}

    /*===== Load all required support files and options =====*/
    private function load_required_files()
    {
        $required_files = [
            'functions/inc/trackngo-cpt.php',
            'functions/inc/trackngo-helper.php',
            'functions/inc/svg-support.php',
            'functions/lib/codestar/codestar-framework.php',

            // Theme options
            'functions/theme-option/csf.php',
            'functions/theme-option/csf-body.php',
            'functions/theme-option/csf-header.php',
            'functions/theme-option/csf-breadcrumb.php',
            'functions/theme-option/csf-typography.php',
            'functions/theme-option/csf-blog.php',
            'functions/theme-option/csf-404.php',
            'functions/theme-option/csf-footer.php',
            'functions/theme-option/csf-custom-css.php',
            'functions/theme-option/csf-backup.php',

            // Metaboxes
            'functions/theme-option/metabox/csf-mt-post.php',
            'functions/theme-option/metabox/csf-mt-page.php',
            'functions/theme-option/metabox/csf-mt-header-footer.php',

            // Widgets
            'functions/wp-widgets/about-author.php',
            'functions/wp-widgets/contact-info.php',
            'functions/wp-widgets/posts.php',
        ];

        // Add Elementor init if Elementor is active
        if (did_action('elementor/loaded')) {
            $required_files[] = 'elementor/elementor-init.php';
        }

        // Load each required file
        foreach ($required_files as $file) {
            $path = $this->dir . ltrim($file, '/');
            if (file_exists($path)) {
                require_once $path;
            } else {
                error_log("[Trackngo-Core] Required file not found: " . esc_html($file));
            }
        }
    }
}


// Initialize plugin
Trackngo_Core_Plugin::instance();


/***** Load widgets Files*****/
function trackngo_load_widget_files()
{

    $widget_files = [
        'functions/wp-widgets/widget-recent-posts.php',
        'functions/wp-widgets/widget-search.php',
        'functions/wp-widgets/widget-taxonomy.php',
        'functions/wp-widgets/widget-post-tags.php',
    ];

    foreach ($widget_files as $file) {
        $path = TRACKNGO_DIR_PATH . $file;

        if (file_exists($path)) {
            require_once $path;
        } else {
            error_log('[Trackngo-Core] Missing widget file: ' . $file);
        }
    }
}
trackngo_load_widget_files();

/***** Register widgets *****/
function trackngo_register_widgets()
{
    $widgets = [
        'Trackngo_Core_Recent_Post_Widget',
        'Trackngo_Core_Search_Widget',
        'Trackngo_Core_Post_Taxonomy',
        'Trackngo_Core_Post_Tags_Widget',
    ];

    foreach ($widgets as $widget) {
        if (class_exists($widget)) {
            register_widget($widget);
        } else {
            error_log("[Trackngo-Core] Widget class not found: {$widget}");
        }
    }
}
add_action('widgets_init', 'trackngo_register_widgets');
